----------------
title : C++ 레퍼런스 - all_of 함수
cat_title : all_of
ref_title : all_of, std::all_of
path : /C++ Reference/algorithm
publish_date : 2019-01-16
----------------

##@ cpp-ref-start

#@ all_of

`<algorithm>` 에 정의됨

```cpp-formatted
template <class InputIterator, class UnaryPredicate>
bool all_of(InputIterator first, InputIterator last, UnaryPredicate pred);
```

만일 범위 내의 (`first` 부터 `last` 전 까지) 원소들에 대해 `pred` 가 모두 `true` 를 리턴한다면 `all_of` 역시 `true` 를 리턴합니다.

한 개라도 `false` 를 리턴한다면 `all_of` 는 `false` 를 리턴합니다. 즉, 모든 원소들에 대해 조건이 맞는지 확인합니다.

### 인자들

* `first, last` : 원소들의 시작점과 끝점을 가리키는 반복자(`InputIterator`)들. 이 때, 확인하는 범위는 `[first, last)` 로 정의됩니다. `first` 가 가리키는 원소는 포함되지만 `last` 가 가리키는 원소는 포함되지 않습니다.

* `pred` : 인자를 한 개만 가지는 함수로 `bool` 로 변환 가능한 타입을 리턴해야 합니다.

### 리턴값

만일 모든 원소들에 대해 `pred` 가 `true` 를 리턴했다면 `true` 를, 아니면 `false` 를 리턴합니다.

### 구현 예시

```cpp-formatted
template <class InputIterator, class UnaryPredicate>
bool all_of(InputIterator first, InputIterator last, UnaryPredicate pred) {
  while (first != last) {
    if (!pred(*first)) return false;
    ++first;
  }
  return true;
}
```

아래는 `std::find_if_not` 을 이용한 방식 입니다. 참고로 `std::find_if_not` 은 조건을 만족하지 않는 첫번째 원소를 리턴합니다.

```cpp-formatted
template <class InputIt, class UnaryPredicate>
constexpr bool all_of(InputIt first, InputIt last, UnaryPredicate p) {
  return std::find_if_not(first, last, p) == last;
}
```

참고로 C++ 17 부터 `bool` 대신 `constexpr bool` 을 리턴합니다.

### 실행 예제

```cpp-formatted
#include <algorithm>  // std::all_of
#include <array>      // std::array
#include <iostream>   // std::cout

int main() {
  std::array<int, 8> foo = {3, 5, 7, 11, 13, 17, 19, 23};

  if (std::all_of(foo.begin(), foo.end(), [](int i) { return i % 2; }))
    std::cout << "모든 원소들이 홀수 입니다\n";

  return 0;
}
```

실행 결과

```exec
모든 원소들이 홀수 입니다
```

### 참고 자료

* `any_of` : 범위 안의 원소들 중 조건을 만족하는 원소가 있는지 확인한다.
* `none_of` : 범위 안에 모든 원소들이 조건을 만족하지 않는지 확인한다.
* `find_if_not` : 범위 안에 원소들 중 조건과 일치하지 않는 원소를 찾는다.